<?php
/**
 * Plugin Name: SiteGuidance Bridge
 * Description: Receives metadata and supports direct onboarding from WordPress.
 * Version: 0.2.0
 * Author: SiteGuidance
 */

if (!defined('ABSPATH')) {
    exit;
}

define('SG_BRIDGE_OPTION_API_BASE', 'sg_bridge_api_base');
define('SG_BRIDGE_OPTION_SITE_TOKEN', 'sg_bridge_site_token');
define('SG_BRIDGE_OPTION_LAST_STATUS', 'sg_bridge_last_status');

function sg_bridge_permission_callback() {
    return current_user_can('edit_posts');
}

function sg_bridge_update_post_meta(WP_REST_Request $request) {
    $post_id = intval($request->get_param('post_id'));
    if (!$post_id || !get_post($post_id)) {
        return new WP_REST_Response(array('ok' => false, 'error' => 'invalid_post_id'), 400);
    }

    $source_task_id = sanitize_text_field((string) $request->get_param('source_task_id'));
    $source_draft_id = sanitize_text_field((string) $request->get_param('source_draft_id'));
    $source_title = sanitize_text_field((string) $request->get_param('source_title'));
    $focus_keyword = sanitize_text_field((string) $request->get_param('focus_keyword'));
    $seo_title = sanitize_text_field((string) $request->get_param('seo_title'));
    $seo_description = sanitize_text_field((string) $request->get_param('seo_description'));

    if ($source_task_id !== '') {
        update_post_meta($post_id, '_sg_source_task_id', $source_task_id);
    }
    if ($source_draft_id !== '') {
        update_post_meta($post_id, '_sg_source_draft_id', $source_draft_id);
    }
    if ($source_title !== '') {
        update_post_meta($post_id, '_sg_source_title', $source_title);
    }
    if ($focus_keyword !== '') {
        update_post_meta($post_id, '_sg_focus_keyword', $focus_keyword);
    }
    if ($seo_title !== '') {
        update_post_meta($post_id, '_sg_seo_title', $seo_title);
    }
    if ($seo_description !== '') {
        update_post_meta($post_id, '_sg_seo_description', $seo_description);
    }

    update_post_meta($post_id, '_sg_last_synced_at', gmdate('c'));

    return new WP_REST_Response(array(
        'ok' => true,
        'post_id' => $post_id,
    ), 200);
}

add_action('rest_api_init', function () {
    register_rest_route('siteguidance/v1', '/post-meta', array(
        'methods' => 'POST',
        'callback' => 'sg_bridge_update_post_meta',
        'permission_callback' => 'sg_bridge_permission_callback',
    ));
});

function sg_bridge_get_api_base() {
    $saved = get_option(SG_BRIDGE_OPTION_API_BASE, 'https://siteguidance.com');
    $base = rtrim(trim((string) $saved), '/');
    if ($base === '') {
        $base = 'https://siteguidance.com';
    }
    return $base;
}

function sg_bridge_mask_token($token) {
    $token = (string) $token;
    if ($token === '') return '';
    if (strlen($token) <= 12) return '************';
    return substr($token, 0, 8) . '...' . substr($token, -4);
}

function sg_bridge_call_api($method, $path, $body = null, $site_token = '') {
    $url = sg_bridge_get_api_base() . $path;
    $args = array(
        'method' => $method,
        'timeout' => 20,
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
    );
    if ($site_token !== '') {
        $args['headers']['Authorization'] = 'Bearer ' . $site_token;
    }
    if ($body !== null) {
        $args['body'] = wp_json_encode($body);
    }

    $response = wp_remote_request($url, $args);
    if (is_wp_error($response)) {
        return array('ok' => false, 'error' => $response->get_error_message(), 'status' => 0);
    }
    $status = wp_remote_retrieve_response_code($response);
    $raw = wp_remote_retrieve_body($response);
    $decoded = json_decode($raw, true);
    if (!is_array($decoded)) {
        $decoded = array('raw' => $raw);
    }
    $decoded['_http_status'] = $status;
    return $decoded;
}

function sg_bridge_admin_handle_register() {
    if (!current_user_can('manage_options')) {
        wp_die('forbidden');
    }
    check_admin_referer('sg_bridge_register');

    $company_name = sanitize_text_field((string) ($_POST['company_name'] ?? ''));
    $contact_name = sanitize_text_field((string) ($_POST['contact_name'] ?? ''));
    $contact_email = sanitize_email((string) ($_POST['contact_email'] ?? ''));
    $wp_url = home_url();
    $domain = wp_parse_url($wp_url, PHP_URL_HOST);

    if ($company_name === '' || $contact_email === '') {
        update_option(SG_BRIDGE_OPTION_LAST_STATUS, 'Missing company/email.');
        wp_safe_redirect(admin_url('options-general.php?page=siteguidance-bridge'));
        exit;
    }

    $result = sg_bridge_call_api('POST', '/public/plugin/register', array(
        'companyName' => $company_name,
        'contactName' => $contact_name,
        'contactEmail' => $contact_email,
        'domain' => $domain,
        'wpUrl' => $wp_url,
    ));

    if (!empty($result['ok']) && !empty($result['siteToken'])) {
        update_option(SG_BRIDGE_OPTION_SITE_TOKEN, sanitize_text_field((string) $result['siteToken']));
        update_option(SG_BRIDGE_OPTION_LAST_STATUS, 'Connected and token issued.');
    } else {
        $error = isset($result['error']) ? (string) $result['error'] : 'register_failed';
        update_option(SG_BRIDGE_OPTION_LAST_STATUS, 'Register failed: ' . $error);
    }

    wp_safe_redirect(admin_url('options-general.php?page=siteguidance-bridge'));
    exit;
}
add_action('admin_post_sg_bridge_register', 'sg_bridge_admin_handle_register');

function sg_bridge_admin_handle_wp_config() {
    if (!current_user_can('manage_options')) {
        wp_die('forbidden');
    }
    check_admin_referer('sg_bridge_wp_config');

    $token = (string) get_option(SG_BRIDGE_OPTION_SITE_TOKEN, '');
    if ($token === '') {
        update_option(SG_BRIDGE_OPTION_LAST_STATUS, 'No site token. Register first.');
        wp_safe_redirect(admin_url('options-general.php?page=siteguidance-bridge'));
        exit;
    }

    $wp_username = sanitize_text_field((string) ($_POST['wp_username'] ?? ''));
    $wp_app_password = sanitize_text_field((string) ($_POST['wp_app_password'] ?? ''));
    $wp_status = sanitize_text_field((string) ($_POST['wp_status'] ?? 'draft'));
    $auto_publish = !empty($_POST['auto_publish']) ? true : false;

    $result = sg_bridge_call_api('POST', '/public/plugin/wordpress-config', array(
        'wpUrl' => home_url(),
        'wpUsername' => $wp_username,
        'wpAppPassword' => $wp_app_password,
        'wpStatus' => $wp_status,
        'autoPublishOnApproval' => $auto_publish,
    ), $token);

    if (!empty($result['ok'])) {
        update_option(SG_BRIDGE_OPTION_LAST_STATUS, 'WordPress config saved to SiteGuidance.');
    } else {
        $error = isset($result['error']) ? (string) $result['error'] : 'wp_config_failed';
        update_option(SG_BRIDGE_OPTION_LAST_STATUS, 'WordPress config failed: ' . $error);
    }

    wp_safe_redirect(admin_url('options-general.php?page=siteguidance-bridge'));
    exit;
}
add_action('admin_post_sg_bridge_wp_config', 'sg_bridge_admin_handle_wp_config');

function sg_bridge_admin_handle_status() {
    if (!current_user_can('manage_options')) {
        wp_die('forbidden');
    }
    check_admin_referer('sg_bridge_status');

    $token = (string) get_option(SG_BRIDGE_OPTION_SITE_TOKEN, '');
    if ($token === '') {
        update_option(SG_BRIDGE_OPTION_LAST_STATUS, 'No site token. Register first.');
        wp_safe_redirect(admin_url('options-general.php?page=siteguidance-bridge'));
        exit;
    }

    $result = sg_bridge_call_api('GET', '/public/plugin/status', null, $token);
    if (!empty($result['ok'])) {
        update_option(SG_BRIDGE_OPTION_LAST_STATUS, 'Connected. Status check OK.');
    } else {
        $error = isset($result['error']) ? (string) $result['error'] : 'status_failed';
        update_option(SG_BRIDGE_OPTION_LAST_STATUS, 'Status failed: ' . $error);
    }

    wp_safe_redirect(admin_url('options-general.php?page=siteguidance-bridge'));
    exit;
}
add_action('admin_post_sg_bridge_status', 'sg_bridge_admin_handle_status');

function sg_bridge_register_admin_page() {
    add_options_page(
        'SiteGuidance Bridge',
        'SiteGuidance Bridge',
        'manage_options',
        'siteguidance-bridge',
        'sg_bridge_render_admin_page'
    );
}
add_action('admin_menu', 'sg_bridge_register_admin_page');

function sg_bridge_render_admin_page() {
    if (!current_user_can('manage_options')) {
        return;
    }
    $api_base = sg_bridge_get_api_base();
    $site_token = (string) get_option(SG_BRIDGE_OPTION_SITE_TOKEN, '');
    $last_status = (string) get_option(SG_BRIDGE_OPTION_LAST_STATUS, 'Ready.');
    ?>
    <div class="wrap">
      <h1>SiteGuidance Bridge</h1>
      <p><strong>API Base:</strong> <?php echo esc_html($api_base); ?></p>
      <p><strong>Site token:</strong> <?php echo esc_html(sg_bridge_mask_token($site_token)); ?></p>
      <p><strong>Last status:</strong> <?php echo esc_html($last_status); ?></p>

      <hr />
      <h2>1) Register / Connect</h2>
      <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <?php wp_nonce_field('sg_bridge_register'); ?>
        <input type="hidden" name="action" value="sg_bridge_register" />
        <table class="form-table">
          <tr><th><label>Company name</label></th><td><input name="company_name" class="regular-text" required /></td></tr>
          <tr><th><label>Contact name</label></th><td><input name="contact_name" class="regular-text" /></td></tr>
          <tr><th><label>Contact email</label></th><td><input name="contact_email" type="email" class="regular-text" required /></td></tr>
        </table>
        <p><button class="button button-primary" type="submit">Register with SiteGuidance</button></p>
      </form>

      <hr />
      <h2>2) Save WordPress Publish Config</h2>
      <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <?php wp_nonce_field('sg_bridge_wp_config'); ?>
        <input type="hidden" name="action" value="sg_bridge_wp_config" />
        <table class="form-table">
          <tr><th><label>WP username</label></th><td><input name="wp_username" class="regular-text" /></td></tr>
          <tr><th><label>WP app password</label></th><td><input name="wp_app_password" class="regular-text" /></td></tr>
          <tr><th><label>Publish status</label></th><td><select name="wp_status"><option value="draft">draft</option><option value="publish">publish</option><option value="pending">pending</option></select></td></tr>
          <tr><th><label>Auto publish on approval</label></th><td><input name="auto_publish" type="checkbox" value="1" /></td></tr>
        </table>
        <p><button class="button button-primary" type="submit">Save Config</button></p>
      </form>

      <p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
          <?php wp_nonce_field('sg_bridge_status'); ?>
          <input type="hidden" name="action" value="sg_bridge_status" />
          <button class="button" type="submit">Check Connection Status</button>
        </form>
      </p>
    </div>
    <?php
}
